﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using UnityEngine;
using UnityEngine.UI;
using UnityEngine.XR;

//CarDefinition is a script that defines a game object as representing a car
public class CarDefinition : MonoBehaviour
{
    //Properties of the car
    public string NamePrefix { get; private set; }
    public string NameSuffix { get; private set; }
    public string Name { get; private set; }
    public int Price { get; private set; }
    public Color Colour { get; private set; }
    public CarController Controller { get; private set; }
    public CarAudio Audio { get; private set; }

    private BoxCollider2D _Collider;

    //Initialize the car with its required data
    public void Initialize(string namePrefix, string nameSuffix, int price, Color colour, Sprite bodySprite, Dictionary<CarPart, Sprite> parts, float maxSpeed, float maxReverseSpeed, float acceleration, float handling, float driftAmount, AudioFile engineSoundFile)
    {
        NamePrefix = namePrefix;
        NameSuffix = nameSuffix;
        Name = NamePrefix + " " + NameSuffix;
        Price = price;
        Colour = colour;
        Controller.MaxSpeed = maxSpeed;
        Controller.MaxReverseSpeed = maxReverseSpeed;
        Controller.Acceleration = acceleration;
        Controller.Handling = handling;
        Controller.DriftAmount = driftAmount;
        Audio.EngineClip = engineSoundFile.Clip;

        SetupCombinedSprite(bodySprite, parts);
    }

    public void Awake()
    {
        Controller = gameObject.GetComponent<CarController>();
        Audio = gameObject.GetComponent<CarAudio>();
        _Collider = gameObject.GetComponent<BoxCollider2D>();
    }

    /// <summary>
    /// Computes the maximum speed into a fractional range
    /// </summary>
    /// <param name="fraction">The upper range of fraction</param>
    /// <returns>The max speed as a value in the fractional range</returns>
    public float GetFractionalMaxSpeed(float fraction = 5.0f)
    {
        float oneFraction = ConfigurationManager.Instance.Cars.HighEnd.SpeedUpperRange / fraction;
        return (float)System.Math.Round(Controller.MaxSpeed / oneFraction, ConfigurationManager.Instance.Cars.StatsDecimalPlaces);
    }

    /// <summary>
    /// Computes the maximum speed as a percentage
    /// </summary>
    /// <returns>The max speed as a percentage</returns>
    public int GetSpeedPercentage()
    {
        float fractionalMaxSpeed = GetFractionalMaxSpeed();
        return Convert.ToInt32((fractionalMaxSpeed / 5.0f) * 100.0f);
    }

    /// <summary>
    /// Computes the acceleration into a fractional range
    /// </summary>
    /// <param name="fraction">The upper range of acceleration</param>
    /// <returns>The acceleration as a value in the fractional range</returns>
    public float GetFractionalAcceleration(float fraction = 5.0f)
    {
        float oneFraction = ConfigurationManager.Instance.Cars.HighEnd.AccelerationUpperRange / fraction;
        return (float)System.Math.Round(Controller.Acceleration / oneFraction, ConfigurationManager.Instance.Cars.StatsDecimalPlaces);
    }

    /// <summary>
    /// Computes the acceleration as a percentage
    /// </summary>
    /// <returns>The acceleration as a percentage</returns>
    public int GetAccelerationPercentage()
    {
        float fractionalAccel = GetFractionalAcceleration();
        return Convert.ToInt32((fractionalAccel / 5.0f) * 100.0f);
    }

    /// <summary>
    /// Computes the handling into a fractional range
    /// </summary>
    /// <param name="fraction">The upper range of handling</param>
    /// <returns>The handling as a value in the fractional range</returns>
    public float GetFractionalHandling(float fraction = 200.0f)
    {
        float oneFraction = ConfigurationManager.Instance.Cars.HighEnd.HandlingUpperRange / fraction;
        return (float)System.Math.Round(Controller.Handling / oneFraction, ConfigurationManager.Instance.Cars.StatsDecimalPlaces);
    }

    /// <summary>
    /// Computes the handling as a percentage
    /// </summary>
    /// <returns>The handling as a percentage</returns>
    public int GetHandlingPercentage()
    {
        float fractionalHandling = GetFractionalHandling();
        return Convert.ToInt32((fractionalHandling / 200.0f) * 100.0f);
    }

    /// <summary>
    /// Combines the body and parts sprites into a single final sprite
    /// </summary>
    /// <param name="bodySprite">The sprite of the car body</param>
    /// <param name="partsSprites">The sprites of all the parts</param>
    private void SetupCombinedSprite(Sprite bodySprite, Dictionary<CarPart, Sprite> partsSprites)
    {
        //Create our combined sprite based on the body
        Texture2D combinedTexture = new Texture2D(Convert.ToInt32(bodySprite.texture.width), Convert.ToInt32(bodySprite.texture.height));
        Texture2D bodyTexture = new Texture2D(Convert.ToInt32(bodySprite.texture.width), Convert.ToInt32(bodySprite.texture.height));

        //The final combined texture colours
        List<Color> textureColours = new List<Color>();

        //First, let's tint the body and add it to the final texture
        Color[] bodyColours = bodySprite.texture.GetPixels();

        for (int i = 0; i < bodyColours.Count(); i++)
        {
            bodyColours[i] *= Colour;
        }

        bodyTexture.SetPixels(bodyColours);
        bodyTexture.Apply();
        textureColours.AddRange(bodyTexture.GetPixels());

        //Now, let's create a texture just containing the parts
        Texture2D partsTexture = new Texture2D(Convert.ToInt32(bodySprite.texture.width), Convert.ToInt32(bodySprite.texture.height));
        List<Color> partsColours = new List<Color>();

        //Let's start off with a fully transparent texture
        for(int i = 0; i < bodySprite.texture.width * bodySprite.texture.height; i++)
        {
            partsColours.Add(Color.clear);
        }

        foreach (var partKeyValPair in partsSprites)
        {
            //Get all the pixels for this part
            Color[] thisPartColours = partKeyValPair.Value.texture.GetPixels();

            if (partKeyValPair.Key.Tintable)
            {
                //And tint it
                for(int i = 0; i < thisPartColours.Count(); i++)
                {
                    thisPartColours[i] *= Colour;
                }
            }

            for(int i = 0; i < thisPartColours.Count(); i++)
            {
                //Now, if our combined parts texture doesn't have an occupied pixel already, let's add it from this part
                if(partsColours[i].a == 0.0f)
                {
                    partsColours[i] = thisPartColours[i];
                }
            }
        }

        //Now we've added all our parts, let's make the final parts texture
        partsTexture.SetPixels(partsColours.ToArray());
        partsTexture.Apply();
        //We can save the parts sprite here

        //Finally, loop through the base body texture, and add on the parts texture colour if one is present
        for (int i = 0; i < bodySprite.texture.width * bodySprite.texture.height; i++)
        {
            if(partsColours[i].a != 0.0f)
            {
                textureColours[i] = partsColours[i];
            }
        }

        //All combined
        combinedTexture.SetPixels(textureColours.ToArray());
        combinedTexture.Apply();
        //We could save the final sprite here

        //Finally, make the combined sprite based on the texture
        Sprite combinedSprite = Sprite.Create(combinedTexture, new Rect(0, 0, combinedTexture.width, combinedTexture.height), new Vector2(0.5f, 0.5f));

        gameObject.GetComponent<SpriteRenderer>().sprite = combinedSprite;
        gameObject.GetComponent<SpriteRenderer>().color = Color.white;
        gameObject.transform.localScale = new Vector3(ConfigurationManager.Instance.Player.Scale, ConfigurationManager.Instance.Player.Scale, 1.0f);

        //And set up the collider
        _Collider.size = new Vector2(combinedSprite.bounds.size.x, combinedSprite.bounds.size.y);
    }
}
